-- For this tutorial, we will include a set of custom functions, that will make the code
-- shorter and increase readibility. The function dofile() simply appends the content of the 
-- specified file at the position where this function is called.
dofile("functions_Fabian.lua")

------------------------------------------------------------------------------------------------------------------------------------------
-- Initialize HFM
------------------------------------------------------------------------------------------------------------------------------------------
-- In this script, we only want ot calculate the Heisenberg ferromagnet with
-- its one magnon and two magnon spectral function. Therefore, we need to define
-- lattice vectors and magnetic exchange constants that describe the hopping of
-- spin excitations along these directions.
lattices_EuO = {    
    {{ 1, 0, 0}},
    {{ 0, 1, 0}},
    {{ 0, 0, 1}},
    {{ 1/2, 1/2, 0}},
    {{ 0, 1/2, 1/2}},
    {{ 1/2, 0, 1/2}}, 
    {{ 1/2,-1/2, 0}},
    {{ 0, 1/2,-1/2}},
    {{-1/2, 0, 1/2}},
}

meV = 0.001
S = 7/2

-- we include the next nearest neighbor exchange along the simple cubic directions.
-- But as we are in a fcc lattice, the nearest neighbors sit at the 12 faces.
J_nn = 2*0.606*phys["kB"]*S/meV
J_nnn = 2*0.119*phys["kB"]*S/meV

-- To map the hoppings to the correct lattice vectors, we need to have the same order.
hoppings_EuO = {J_nnn, J_nnn, J_nnn, J_nn, J_nn, J_nn, J_nn, J_nn, J_nn}
----------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------


------------------------------------------------------------------------------------------------------------------------------------------
--Initialize calculation
------------------------------------------------------------------------------------------------------------------------------------------
NPoints = 100
NE = 100
Emin = 0
Emax = 12
gamma = 0.5
-- The Brillouin zone boundary is at 2pi
BZ = math.pi*2
Nq = 30
----------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------

-- To capture all features of this Heisenberg system, we move along paths connecting the
-- pricipal lattice points in a cubic Brillouin zone.
G = {{0, 0, 0}}
X = {{BZ, 0, 0}}
M = {{BZ, BZ, 0}}
R = {{BZ, BZ, BZ}}

-- to use Matrix operations on the k-points specified above, we set the repsective meta table.
-- This concept is essential in Lua and needs to be understood.
setmetatable(G, MatrixMeta)
setmetatable(X, MatrixMeta)
setmetatable(M, MatrixMeta)
setmetatable(R, MatrixMeta)

pathGX = (X - G)
pathXG = (G - X)

pathXM = (M - X)
pathMG = (G - M)

pathMR = (R - M)
pathRG = (G - R)

paths = {pathGX, pathXM, pathMR, pathRG}

-- we set an output file for the spectral funcitons
OUT_w_k = io.open(directory .. "HFM_S_w_k_EuO.dat", "w")

-- we start at gamma
k0 = G

NPoints = NPoints/#paths

TimeStart("Calculation")

-- we iterator over all paths
for i, p in pairs(paths) do
    
    -- then we itereate over all k points per path
    for j = 0, NPoints - 1 do
        
        k = k0 + p*j/NPoints
        
        -- this funciton calculates the single magnon dispersion.
        local w_k = HFM_w_k(k, hoppings_EuO, lattices_EuO)
    
        -- for the spectral functions, we need to iterate over an energy range
        for e = 0, NE do
            
            local w = Emin + (Emax - Emin)/NE*e
            
            -- the output file will have 4 columns, k, w, S1_w_k and S2_w_k
            OUT_w_k:write(string.format("%.15e ", j + (i - 1)*NPoints))
            OUT_w_k:write(string.format("%.15e ", w))
            
            -- this funciton calculates the single magnon spectral function. As we only have a single resonance
            -- this in principle is only a Lorenzian at w_k
            OUT_w_k:write(string.format("%.15e ", HFM_S1_w_k(w, k, hoppings_EuO, lattices_EuO, gamma, w_k)))
            
            -- For the two magnon spectral function, instead, we will observe a broad band.
            -- Exciting two fundamental particles at a particular momentum k leaves many possibilities
            -- which single particle momenta will add up to the total momentum k. Both particles could have different energies.
            -- Integrating over all possible combinations (here discretized by the grid points Nq for kx, ky, kz)
            -- will lead to a broad band instead of a single resonance. 
            -- This will take a LOT OF TIME!
            OUT_w_k:write(string.format("%.15e ", HFM_S2_w_k(w, k, hoppings_EuO, lattices_EuO, gamma, Nq, Nq, Nq)))
            OUT_w_k:write("\n")
        end
        
        print(j + (i - 1)*NPoints)
        
        OUT_w_k:write("\n")
    end

    k0 = k0 + p
end

TimeEnd("Calculation")

OUT_w_k:close()

TimePrint()
